********************************************************************************
* STEP 1: GENERATE VARIABLES FOR ANALYSIS (in Step 2)
* 
*   1. Preparatory steps (setup and load raw dataset)
*   2. Generate variable on presence of children
*   3. Code actual division of household tasks
*   4. Generate winsorized inflation questions
*   5. Get dummies for product categories
*   6. Code financial literacy measure
*   7. Drop observations with non-binary gender
*   8. Save data in new dataset (infl_gender.dta)
********************************************************************************


/*====================================================================
   1. Preparatory steps
  ====================================================================*/

    // standard setup
    clear all
    set maxvar 30000
    set more off
    
    // set paths
    global path    = "PATH/TO/YOUR/FOLDER"
    global data    = "$path/dta"
    global results = "$path/results"
    
    // load the raw dataset
    use "$data/infl_raw.dta", clear
    
    
/*====================================================================
   2. Generate variable on presence of children
  ====================================================================*/
        
    // children present
    gen children = .
    replace children = 0 if q9 == 2
    replace children = 1 if q9 == 1
    
    // labeling
    label define children ///
            0  "No" ///
            1  "Yes"
    label values children children
    label variable children "Children present"
    
    
/*====================================================================
   3. Code actual division of household tasks
     - traditional for men: "shared with someone else" or "Someone else primarily responsible "
     - traditional for women: "I am primarily responsible"
  ====================================================================*/
    
    // traditional household organization recoding
    foreach v of varlist q16_1 q16_2 q16_3 {
        gen `v'_trad = .
        replace `v'_trad = 0 if !mi(`v')
        replace `v'_trad = 1 if q1 == 1 & `v' > 1 & !mi(`v')
        replace `v'_trad = 1 if q1 == 2 & `v' == 1 & !mi(`v')
    }
    
    // calculate a traditional behavior component using pca
    pca q16_1_trad q16_2_trad q16_3_trad, component(1)
    
    // store the factor in one variable
    predict q16_temp, score
    
    // now create a 0/1 variables (this is the same as saying it is 1 if any of q16_1 or q16_3 is equal to 1)
    gen q16_trad = .
    replace q16_trad = 0 if q16_temp <= 0 & !mi(q16_temp)
    replace q16_trad = 1 if q16_temp >  0 & !mi(q16_temp)
    drop q16_temp
    
    // labeling
    label variable q16_1_trad "Traditional: food"
    label variable q16_2_trad "Traditional: leisure activities"
    label variable q16_3_trad "Traditional: sick kids"
    label variable q16_trad "Traditional household organization"

    
/*====================================================================
   4. Generate winsorized inflation questions
  ====================================================================*/
    
    foreach v of varlist q18 q19 q20 {
        winsor2 `v', cuts(1 99) suffix(_win)        
    }

    // Inflation expectation
    local var = "q18_win"
    label variable `var' "Inflation expectation (next 12 months, winsorized)"
    
    // Inflation expectation (lower bound)
    local var = "q19_win"
    label variable `var' "Inflation expectation (lower bound, winsorized)"
    
    // Inflation expectation (upper bound)
    local var = "q20_win"
    label variable `var' "Inflation expectation (upper bound, winsorized)"
    
    
/*====================================================================
   5. Get dummies for product categories
  ====================================================================*/
    
    // generate dummy variables for saying product is important
    foreach v of varlist q21_1-q21_10 {
        gen `v'_dummy = 0
        replace `v'_dummy = 1 if `v' == 1
    }
	
	// reorganize categories to match product groups more closely
	gen temp = q21_7_dummy
	replace q21_7_dummy = q21_9_dummy
	replace q21_9_dummy = temp
	drop temp
	
	gen temp = q21_8_dummy
	replace q21_8_dummy = q21_10_dummy
	replace q21_10_dummy = temp
	drop temp
	
    
    // generate variable labels
    label variable q21_1_dummy "Eggs"
    label variable q21_2_dummy "Fresh food/Veggies"
    label variable q21_3_dummy "Dairy/Milk"
    label variable q21_4_dummy "Meat/Poultry"
    label variable q21_5_dummy "Bread"
    label variable q21_6_dummy "Rent"
    label variable q21_7_dummy "Clothing"
    label variable q21_8_dummy "Travel/Flight prices"
    label variable q21_9_dummy "Gasoline/Fuel/Petroleum"
    label variable q21_10_dummy "Heating"
    
    // generate average over three product groups
    egen q21_food     = rowmean(q21_1_dummy-q21_5_dummy)
    egen q21_expenses = rowmean(q21_6_dummy-q21_8_dummy)
    egen q21_energy   = rowmean(q21_9_dummy-q21_10_dummy)
    
    // labeling
    label variable q21_food     "Information sources: food"
    label variable q21_expenses "Information sources: family expenses"
    label variable q21_energy   "Information sources: energy"
    
    
/*====================================================================
   6. Code financial literacy measure
  ====================================================================*/
    
    // number of correct questions
    gen f1 = 0
    replace f1 = 1 if q25 == 2
    gen f2 = 0
    replace f2 = 1 if q26 == 10
    gen fin_lit = f1 + f2
    
    // labeling
    label define fin_lit ///
            0  "Low" ///
            1  "Medium" ///
            2  "High"
    label values fin_lit fin_lit
    label variable fin_lit "Financial literacy"
    
    
/*====================================================================
   7. Drop observations with non-binary gender
  ====================================================================*/
    
    keep if q1 < 3
    
    
/*====================================================================
   8. Save data in new dataset
  ====================================================================*/
    
    save "$data/infl_gender.dta", replace
    
