********************************************************************************
* STEP 2: GENERATE FIGURES AND TABLES FOR PAPER AND ADDITIONAL MATERIALS
* 
*   Preparatory steps (setup, paths, colors, controls)
*   FIGURE 1: Map of differences in inflation expectations
*   FIGURE 2a: Share of individuals in traditional households and inflation expectations
*   FIGURE 2b: Household organization and inflation expectations
*   TABLE  1 + S2 + S3: Gender dummy and family organization
*   FIGURE 3 + TABLE S4: Important product categories
*   FIGURE 4 + TABLE S5: Relation between traditional norms and attention to product categories
*   FIGURE S1: Map of inflation expectations 
*   FIGURE S2: Map of gender difference in inflation expectations (relative)
*   FIGURE S3: Map of women with children in traditional households
*   TABLE  S1: Descriptive statistics
*   TABLE  S6: Alternative explanations
********************************************************************************


/*====================================================================
   Preparatory steps
  ====================================================================*/

// standard setup
    clear all
    set maxvar 30000
    set more off
    
    // set paths
    global path    = "PATH/TO/YOUR/FOLDER"
    global data    = "$path/dta"
    global results = "$path/results"
    global map     = "$path/world_map"
    
    // set graph scheme    
    adopath + "$path"
    set scheme fabian_graphs
    
    // determine colors
    global color1 = "0 77 153"
    global color2 = "234 128 28"
    
    global color3 = "160 160 160"
    global color4 = "41 140 140"
    global color5 = "160 0 0"
    
    // define controls
    global controls  = "i.q3 i.q2 i.q4_group i.q5 i.q7 i.q11 i.q10 ib1.fin_lit i.country_num"
    global controls2 = "i.q3 i.q2 i.q4_group i.q5 i.q7 i.q11 i.q10 i.country_num"
	
	// geoplot package required; install if not already installed
	* ssc install geoplot
    
    
    
/*====================================================================
   FIGURE 1: Map of differences in inflation expectations
  ====================================================================*/
    
    // use dataset
    use "$data/infl_gender.dta", clear
    
    // code inflation questions into different variables by gender
    gen q18_women = q18_win if q1 == 2
    gen q18_men   = q18_win if q1 == 1
    
    // collapse and save
    collapse q18_women q18_men, by(country)
    gen interest = q18_women - q18_men
    rename country ISO_A3_EH
    merge 1:m ISO_A3_EH using "$map/world.dta", keepusing(_ID)
    save "$data/temp.dta", replace
    
    // draw figure
    geoframe create "$map/countries", replace
    geoframe create "$data/temp.dta", replace nodescribe current
    frame countries: geoframe copy temp interest, id(_ID)
    geoplot ///
        (area countries interest, color(viridis, reverse) cuts(0.5 1.5 2.5 3 3.5 5.5) ///
         lc(gs8) lw(0.1) missing(color(gs15) nolabel)) ///
        (area countries interest if _ID == 141, lc(gs8) lw(0.1) missing(color(gs15) nolabel)), ///
        tight legend(position(7) size(3) rowgap(1) region(color(gs15))) ///
        grid(tight lw(0.08) lc(gs10)) project(robinson)
		
	// export figure
    graph export "$results/figure_1.png", replace
    
    
/*====================================================================
   FIGURE 2a: Share of individuals in traditional households and inflation expectations
  ====================================================================*/

    // use dataset
    use "$data/infl_gender.dta", clear    
    
	// code relevant questions
    gen y_var = q18_win
    gen x_var = q16_trad
        		
	// collapse and reshape
    collapse (mean) y_var (mean) x_var, by(q1 country)    
    reshape wide y_var x_var, i(country) j(q1)
    gen diff = y_var2 - y_var1
    gen share = x_var2
    
	// get regression results
    reg diff share, beta
    local t = _b[share]/_se[share]
    local p =2*ttail(e(df_r),abs(`t'))
    local rho = trim("`: display %10.2f e(beta)[1,1]'")
    local p = trim("`: display %10.4f `p''")
    
	// draw figure
    tw  (lfit diff share, color(cranberry) lwidth(medthick) range(0.4 0.8)) ///
        (scatter diff share, mlabel(country) color("$color1") mcolor("$color1") mlabcolor("$color1")), ///
        xlabel(0.4(0.1)0.8) ylabel(0(1)5) ///
        xtitle("Share of women with children in traditional households") ytitle("F-M-Difference Inflation Expectations (in pp)") ///
        legend(label(2 "Data") label(1 "Regression line") order(2 1)) ///
        text(4.9 0.5 "Correlation {it:{&rho} = `rho'} (p-value: {it:`p'})}")
		
	// export figure
    graph export "$results/figure_2a.pdf", replace
    
    
/*====================================================================
   FIGURE 2b: Household organization and inflation expectations
  ====================================================================*/

    // use dataset
    use "$data/infl_gender.dta", clear    
    
	// generate group variables
    gen group = q16_trad+0.15
    replace group = -1 if mi(group)
    
    // collapse data and generate standard errors
    collapse (mean) q18_win (semean) SE = q18_win (count) N = q18_win, by(group q1)
    generate ci_high = q18_win + invttail(N-1, 0.025)*SE
    generate ci_low  = q18_win - invttail(N-1, 0.025)*SE
    
    // adjust for men and women
    replace group = group-0.2 if q1 == 2
    replace group = group+0.2 if q1 == 1
    
    // draw figure
    tw (bar q18_win group if q1 == 2, color("$color1") fintensity(60) barwidth(0.4) lwidth(medthick)) ///
       (bar q18_win group if q1 == 1, color("$color2") fintensity(60) barwidth(0.4) lwidth(medthick)) ///
       (rcap ci_high ci_low group if q1 == 2, lcolor("$color1") msize(huge) lwidth(medthick)) ///
       (rcap ci_high ci_low group if q1 == 1, lcolor("$color2") msize(huge) lwidth(medthick)) , ///
        xlabel(-1 `" "no children" "present" "' 0.15 `" "with children:" "modern roles" "' 1.15 `" "with children:" "traditional roles" "') ///
        xline(-0.425, lcolor(black) lwidth(thin)) ///
        ylabel(10(1)18) yscale(range(9.6 18))  ///
        xtitle(" ") ytitle("Inflation expectation (in %)") ///
        legend(label(1 "Women") label(2 "Men") order(1 2))
		
	// export figure
    graph export "$results/figure_2b.pdf", replace
    
    
/*====================================================================
   TABLE 1 + S2 + S3: Gender dummy and family organization
  ====================================================================*/
  
    // use dataset
    use "$data/infl_gender.dta", clear
	
	// generate variable for organization of family
    gen orga = .
    label define orga ///
            0  "Modern" ///
            1  "Traditional"
    label values orga orga
    label variable orga "Organization of family life"        
        
    // baseline regression only dummy and country fixed effects
    reg q18_win i.q1 i.country_num, robust
    eststo R1
    
    // add financial literacy
    reg q18_win i.q1##i.children i.country_num , robust
    eststo R2
    
    // add controls
    reg q18_win i.q1##i.children $controls, robust
    eststo R3
    
    // only people with children: organization of the family : cooking
    replace orga = .
    replace orga = q16_1_trad
    reg q18_win i.q1##ib0.orga $controls if !mi(q16_1_trad), robust
    eststo R4
    
    // only people with children: organization of the family: leisure activities
    replace orga = .
    replace orga = q16_2_trad
    reg q18_win i.q1##ib0.orga $controls if !mi(q16_2_trad), robust
    eststo R5
    
    // only people with children: organization of the family: staying home when children sick
    replace orga = .
    replace orga = q16_3_trad
    reg q18_win i.q1##ib0.orga $controls if !mi(q16_3_trad), robust
    eststo R6
    
    // only people with children: organization of the family: combined
    replace orga = .
    replace orga = q16_trad
    reg q18_win i.q1##ib0.orga $controls if !mi(q16_3_trad), robust
    eststo R7
    	
	// write regression table files
    esttab R1 R2 R3 R4 R5 R6 R7 using "$results/table_1.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted nobaselevels nofloat nogaps varwidth(50) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 1.children 2.q1#1.children 1.orga 2.q1#1.orga) ///
                 order(2.q1 1.children 2.q1#1.children 1.orga 2.q1#1.orga) ///
                 mlabels("(1)" "(2)" "(3)" "(4)" "(5)" "(6)" "(7)", prefix("{") suffix("}")) ///
                 indicate("Detailed controls= 2.q3" "Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))
    
    
    esttab R1 R2 R3 using "$results/table_S2.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted nobaselevels nofloat nogaps wide varwidth(50) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 1.children 2.q1#1.children *.q3 *.q4_group *.q5 *.q2 *.q7 *.q11 *.q10 *.fin_lit) ///
                 order(2.q1 1.children 2.q1#1.children *.q3 *.q4_group *.q5 *.q2 *.q7 *.q11 *.q10 *.fin_lit) ///
                 mlabels("(1)" "(2)" "(3)", prefix("{") suffix("}")) ///
                 indicate("Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))
                 
                 
    esttab R4 R5 R6 R7 using "$results/table_S3.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted nobaselevels nofloat nogaps wide varwidth(50) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 1.orga 2.q1#1.orga *.q3 *.q2 *.q4_group *.q5 *.q7 *.q11 *.q10 *.fin_lit) ///
                 order(2.q1 1.orga 2.q1#1.orga *.q3 *.q4_group *.q5 *.q2 *.q7 *.q11 *.q10 *.fin_lit) ///
                 mlabels( "(4)" "(5)" "(6)" "(7)", prefix("{") suffix("}")) ///
                 indicate("Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))

                 
    
/*====================================================================
   FIGURE 3 + TABLE S4: Important product categories
  ====================================================================*/
    
    // use dataset
    use "$data/infl_gender.dta", clear
        
    // do regressions
    reg q18_win i.q1 $controls, robust
    eststo R1
    
    reg q18_win i.q1 i.q1#i.(q21_1_dummy-q21_10_dummy) $controls, robust
    eststo R2
    
    // write regression table file
    esttab R1 R2 using "$results/table_S4.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted baselevels nofloat nogaps wide varwidth(45) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 *.q1#1.q21_*_dummy) ///
                 mlabels("(1)" "(2)" , prefix("{") suffix("}")) ///
                 indicate("Detailed controls= 2.q3" "Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))
    
    
    // copy regression results into new dataset
    clear
    
    gen gender = .
    gen category = .
    gen infl_exp = .
    gen stderr = .
    
    insobs 1
    replace gender = 1 if mi(gender)
    replace category = 0 if mi(category)
    replace infl_exp = _b[2.q1] if mi(infl_exp)
    replace stderr = _se[2.q1] if mi(stderr)
    
    forvalues g = 1/2 {
        forvalues c = 1/10 {
            insobs 1
            replace gender = `g' if mi(gender)
            replace category = `c' if mi(category)
            replace infl_exp = _b[`g'.q1#1.q21_`c'_dummy] if mi(infl_exp)
            replace stderr = _se[`g'.q1#1.q21_`c'_dummy] if mi(stderr)
        }
    }
    
    // get error bounds
    generate ci_high = infl_exp + invttail(`e(N)'-`e(df_m)', 0.025)*stderr
    generate ci_low  = infl_exp - invttail(`e(N)'-`e(df_m)', 0.025)*stderr
    
	// label categories
    label define categories ///
            0    "Female" ///
            1    "Eggs" ///
            2    "Fresh food/Veggies" ///
            3    "Dairy/Milk" ///
            4    "Meat/Poultry" ///
            5    "Bread" ///
            6    "Rent" ///
            7    "Clothing" ///
            8    "Travel/Flight prices" ///
            9    "Gasoline/Fuel/Petroleum" ///
           10    "Heating"
    label values category categories
    label variable category "Product categories"
    
    label define gender ///
            1  "Male" ///
            2  "Female"
    label values gender gender
    label variable gender "Gender"
    
	// readjust for gender
    drop if category == 0
    gen xvals = .
    replace xvals = category-0.2 if gender == 2
    replace xvals = category+0.2 if gender == 1
    label values xvals categories    
    replace xvals = xvals + 0.5 if category > 5
    replace xvals = xvals + 0.5 if category > 8
    
	// draw figure
    tw (bar infl_exp xvals if gender == 2, color("$color1") fintensity(60) barwidth(0.4) lwidth(medthick)) ///
       (bar infl_exp xvals if gender == 1, color("$color2") fintensity(60) barwidth(0.4) lwidth(medthick)) ///
       (scatteri 1.7 0.5 1.7 5.5, recast(line) lw(medium) mc(none) lc(gs7) lp(solid)) ///
       (scatteri 1.7 6.0 1.7 9.0, recast(line) lw(medium) mc(none) lc(gs7) lp(solid)) ///
       (scatteri 1.7 9.5 1.7 11.5, recast(line) lw(medium) mc(none) lc(gs7) lp(solid)), ///
        xlabel(1 "Eggs" 2 "Fresh food/Veggies"    3 "Dairy/Milk" 4 "Meat/Poultry" 5 "Bread" 6.5 "Rent" 7.5 "Clothing" 8.5 "Travel/Flight prices" 10 "Gasoline/Fuel/Petroleum" 11 "Heating", angle(30)) ylabel(-0.5(0.5)2.25) /// 
        xtitle("") ytitle("Change in inflation expectation (in %)") ///
        legend(label(1 "Women") label(2 "Men") order(1 2) position(2) ring(0)) ///
        text(1.75 3.0 "Food", color(gs7) placement(north)) ///
        text(1.75 7.5 "Family Expenses", color(gs7) placement(north)) ///
        text(1.75 10.5 "Energy", color(gs7) placement(north))
    
	// export figure
	graph display, xsize(*1.2)
    graph export "$results/figure_3.pdf", replace
    
    
    
/*====================================================================
   FIGURE 4 + TABLE S5: Relation between traditional norms and attention to product categories
  ====================================================================*/
    
	// use dataset
    use "$data/infl_gender.dta", clear
    	
	// calculate group
    gen group = q16_trad+1
    replace group = 0 if mi(group)
    
	// get label
    label define group ///
            0    "no children" ///
            1    "with children: modern" ///
            2    "with children: traditional"
    label values group group
    label variable group "Family type"
    
    // run regressions and store coefficients in matrix to generate graph
    matrix define B = J(3, 3, 0)
    local i = 1
    foreach c of newlist food expenses energy {
      reg q21_`c' ib1.q1##ib0.group $controls, robust
      eststo R`i'
      matrix B[`i', 1] = _b[2.q1]*100
      matrix B[`i', 2] = (_b[2.q1]+_b[2.q1#1.group])*100
      matrix B[`i', 3] = (_b[2.q1]+_b[2.q1#2.group])*100
      local i = `i' + 1
    }
    
    // write regression table file
    esttab R1 R2 R3 using "$results/table_S5.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted nobaselevels nofloat nogaps wide varwidth(45) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 *.group *.q1#*.group _cons) ///
                 mlabels("(1)" "(2)" "(3)" , prefix("{") suffix("}")) ///
                 indicate("Detailed controls= 2.q3" "Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))
            
    // copy regression results into new dataset
    clear
    
    gen family = .
    gen category = .
    gen attention = .
    
    forvalues t = 1/3 {
        forvalues c = 1/3 {
            insobs 1
            replace family = `t' if mi(family)
            replace category = `c' if mi(category)
            replace attention = B[`c', `t'] if mi(attention)
        }
    }
    
    // get labels for categories
    label define categories ///
            1  "Food" ///
            2  "Family expenses" ///
            3  "Energy"
    label values category categories
    label variable category "Product categories"
    
	// get labels for family types
    label define family ///
            0  "No Children" ///
            1  "Children: Modern" ///
            2  "Children: Traditional"
    label values family family
    label variable family "Organization of family life"
    
	// readjust for gender
    drop if category == 0
    gen xvals = .
    replace xvals = category-0.25 if family == 1
    replace xvals = category      if family == 2
    replace xvals = category+0.25 if family == 3
    label values xvals categories
    
    // draw figure
    tw (bar attention xvals if family == 1, color("$color3") fintensity(60) barwidth(0.25) lwidth(medthick)) ///
       (bar attention xvals if family == 2, color("$color4") fintensity(60) barwidth(0.25) lwidth(medthick)) ///
       (bar attention xvals if family == 3, color("$color5") fintensity(60) barwidth(0.25) lwidth(medthick)), ///
        xlabel(1(1)3, valuelabel) ylabel(-2(1)5) /// 
        xtitle("") ytitle("{&Delta}F-M in attention to product groups (in %)", height(5)) ///
        yline(0, lcolor(black) lwidth(thin)) ///
        legend(label(1 "No Children") label(2 "Children: Modern") label(3 "Children: Traditional") order(1 2 3) rows(1))
    
	// export figure
	graph display, xsize(*1.2)
    graph export "$results/figure_4.pdf", replace

    
    
/*====================================================================
   ADDITIONAL FIGURES AND TABLES
  ====================================================================*/

  
/*====================================================================
   FIGURE S1: Map of inflation expectations 
  ====================================================================*/

    // use dataset
    use "$data/infl_gender.dta", clear
	
	// get variable of interest
    rename q18_win interest
    
    // collapse and save
    collapse interest, by(country)
    rename country ISO_A3_EH
    merge 1:m ISO_A3_EH using "$map/world.dta", keepusing(_ID)
    save "$data/temp.dta", replace
    
    // draw figure
    geoframe create "$map/countries", replace
    geoframe create "$data/temp.dta", replace nodescribe current
    frame countries: geoframe copy temp interest, id(_ID)
    geoplot ///
        (area countries interest, color(viridis, reverse) cuts(8 9 10 11 12 13 14 15 16 35) ///
         lc(gs8) lw(0.1) missing(color(gs15) nolabel)) ///
        (area countries interest if _ID == 141, lc(gs8) lw(0.1) missing(color(gs15) nolabel)), ///
        tight legend(position(7) size(3) rowgap(1) region(color(gs15))) ///
        grid(tight lw(0.08) lc(gs10)) project(robinson)
		
    // export figure
    graph export "$results/figure_S1.png", replace
    
    
/*====================================================================
   FIGURE S2: Map of gender difference in inflation expectations (relative)
  ====================================================================*/

    // use data
    use "$data/infl_gender.dta", clear
	
	// get variable of interest
    gen q18_women = q18_win if q1 == 2
    gen q18_men = q18_win if q1 == 1
    
    // collapse and save
    collapse q18_women q18_men q18_win, by(country)
    gen interest = (q18_women - q18_men)/q18_win*100
    rename country ISO_A3_EH
    merge 1:m ISO_A3_EH using "$map/world.dta", keepusing(_ID)
    save "$data/temp.dta", replace
    
    // draw figure
    geoframe create "$map/countries", replace
    geoframe create "$data/temp.dta", replace nodescribe current
    frame countries: geoframe copy temp interest, id(_ID)
    geoplot ///
        (area countries interest, color(viridis, reverse) cuts(8 11 15 17 20 25 30 40 45) ///
         lc(gs8) lw(0.1) missing(color(gs15) nolabel)) ///
        (area countries interest if _ID == 141, lc(gs8) lw(0.1) missing(color(gs15) nolabel)), ///
        tight legend(position(7) size(3) rowgap(1) region(color(gs15))) ///
        grid(tight lw(0.08) lc(gs10)) project(robinson)
	
	// export figure
    graph export "$results/figure_S2.png", replace   
	
	
/*====================================================================
   FIGURE S3: Map of women with children in traditional households
  ====================================================================*/

    // use dataset
    use "$data/infl_gender.dta", clear
	
	// get variable of interest
	rename q16_trad interest
	keep if q1 == 2 & !mi(interest)
	
	// collapse and save
	collapse interest, by(country)
	rename country ISO_A3_EH
	merge 1:m ISO_A3_EH using "$map/world.dta", keepusing(_ID)
	save "$data/temp.dta", replace
	
	// draw figure
	geoframe create "$map/countries", replace
	geoframe create "$data/temp.dta", replace nodescribe current
	frame countries: geoframe copy temp interest, id(_ID)
	geoplot ///
        (area countries interest, color(viridis, reverse) cuts(0.4 0.5 0.55 0.6 0.65 0.7 0.75 0.8) ///
	     lc(gs8) lw(0.1) missing(color(gs15) nolabel)) ///
	    (area countries interest if _ID == 141, lc(gs8) lw(0.1) missing(color(gs15) nolabel)), ///
        tight legend(position(7) size(3) rowgap(1) region(color(gs15))) ///
        grid(tight lw(0.08) lc(gs10)) project(robinson)
		
	// export figure
	graph export "$results/figure_S3.png", replace
    
    
/*====================================================================
   TABLE S1: Descriptive statistics
  ====================================================================*/
  
    // use data
    use "$data/infl_gender.dta", clear
	
	// generate gender specific variables
    gen q18_m = q18_win if q1 == 1
    gen q18_w = q18_win if q1 == 2
	
	// collapse to country dataset
    collapse q18_win q18_w q18_m, by(country)
    gen q18_diff = q18_w - q18_m
    gen q18_rel  = (q18_w - q18_m)/q18_win*100
    
	// write table to screen
    table country, stat(mean q18_win q18_w q18_m q18_diff q18_rel) nformat("%10.3f")
    
    
/*====================================================================
   TABLE S6: Alternative explanations
  ====================================================================*/
  
	// use dataset
    use "$data/infl_gender.dta", clear
    
	// determine confidence measure
    gen confidence = q20_win - q19_win    
    
    // baseline regression with confidence
    reg q18_win i.q1 $controls2, robust
    eststo R1
    
    // financial literacy with controls
    reg q18_win i.q1 i.q1#ib1.fin_lit $controls2, robust
    eststo R2
    
    // + forecast confidence
    reg q18_win i.q1 i.q1#ib1.fin_lit i.q1#c.confidence  $controls2, robust
    eststo R3
    
    // plus product categories
    reg q18_win i.q1 i.q1#ib1.fin_lit i.q1#c.confidence  i.q1#i.(q21_1_dummy-q21_10_dummy) $controls2, robust
    eststo R4
    
    // write regression table file
    esttab R1 R2 R3 R4 using "$results/table_S6.tex", replace ///
                 se(%7.3f) b(%7.3f) r2 compress label nonotes nonumbers nomtitles noomitted baselevels nofloat nogaps varwidth(45) ///
                 star(* 0.10 ** 0.05 *** 0.01) ///
                 keep(2.q1 *.q1#0.fin_lit  *.q1#2.fin_lit *.q1#c.confidence) ///
                 mlabels("(1)" "(2)" "(3)" "(4)" "(5)" , prefix("{") suffix("}")) ///
                 indicate("Product categories=*.q21_1_dummy" "Detailed controls= 2.q3" "Country FE= 2.country_num" , labels("{Yes}" "{No}")) ///
                 stats(N r2, layout({@} {@}) fmt(a3 %5.3f) labels("Number of Cases" "R-Square"))
